package com.afanticar.afantiopenapi.utils;

import org.apache.commons.lang3.time.DateFormatUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Date;

/**
 * 日期处理类
 *SimpleDateFormat 不是线程安全的，多线程时可能会抛出
 * java.lang.NumberFormatException: multiple points
 */
public class DateUtils extends org.apache.commons.lang3.time.DateUtils{

    private static final Logger LOGGER = LoggerFactory.getLogger(DateUtils.class);

    private static String today;

    /** default date format */
    public final static SimpleDateFormat SDF_YYYY_MM_DD_HH_MM_SS = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
    public final static SimpleDateFormat SDF_YYYY_MM_DD_HH_MM = new SimpleDateFormat("yyyy-MM-dd HH:mm");
    public final static SimpleDateFormat SDF_YYYY_MM_DD = new SimpleDateFormat("yyyy-MM-dd");
    public final static SimpleDateFormat SDF_YYYY_MM = new SimpleDateFormat("yyyy-MM");
    public final static SimpleDateFormat SDF_MM_DD = new SimpleDateFormat("MM-dd");
    public final static SimpleDateFormat SDF_YYYY_MM_DD_CN = new SimpleDateFormat("yyyy年MM月dd日");
    public final static SimpleDateFormat SDF_YYYY_MM_DD_HH_MM_SS_CN = new SimpleDateFormat("yyyy年MM月dd日 HH:mm:ss");
    public final static SimpleDateFormat SDF_YYYY_MM_CN = new SimpleDateFormat("yyyy年MM月");
    public final static SimpleDateFormat SDF_MM_DD_CN = new SimpleDateFormat("MM月dd日");
    public final static SimpleDateFormat SDF_HH_MM_SS = new SimpleDateFormat("HH:mm:ss");
    public final static SimpleDateFormat SDF_YYYYMMDDHHMMSS = new SimpleDateFormat("yyyyMMddHHmmss");

    public static String format(Date date) {
        return DateFormatUtils.format(date, "yyyy-MM-dd HH:mm:ss");
        /*String dateStr = null;
        synchronized(SDF_YYYY_MM_DD_HH_MM_SS) {
            dateStr = SDF_YYYY_MM_DD_HH_MM_SS.format(date);
        }
        return dateStr;*/
    }

    public static String format(Date date, SimpleDateFormat sdf) {

        String dateStr = null;
        synchronized(sdf) {
            dateStr = sdf.format(date);
        }
        return dateStr;
    }

    public static String format(Long date) {
        return DateFormatUtils.format(date, "yyyy-MM-dd HH:mm:ss");
    }

    public static Date parse(String dateStr) {

        Date date = null;
        synchronized(SDF_YYYY_MM_DD_HH_MM_SS) {
            try {
                date = SDF_YYYY_MM_DD_HH_MM_SS.parse(dateStr);
            } catch (ParseException e) {
                LOGGER.warn("Date parse error. dateStr = {}", dateStr);
            }
        }
        return date;
    }

    public static Date parseDate(String dateStr){
        try {
            return org.apache.commons.lang3.time.DateUtils.parseDate(dateStr,"yyyy-MM-dd HH:mm:ss");
        } catch (ParseException e) {
            e.printStackTrace();
        }
        return null;
    }

    public static Date parseDate(String dateStr, String... parsePatterns){
        try {
            return org.apache.commons.lang3.time.DateUtils.parseDate(dateStr,parsePatterns);
        } catch (ParseException e) {
            e.printStackTrace();
        }
        return null;
    }


    public static Date parse(String dateStr, SimpleDateFormat sdf) {

        Date date = null;
        synchronized(sdf) {
            try {
                date = sdf.parse(dateStr);
            } catch (ParseException e) {
                LOGGER.warn("Date parse error. dateStr = {}", dateStr);
            }
        }
        return date;
    }

    public static Date getLastDay() {
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(new Date());
        calendar.add(Calendar.DATE, -1);
        return calendar.getTime();
    }

    /**
     * 今天日期 yyyy-MM-dd
     * @return
     */
    public static String getToday() {
//        if (today == null) {
//            today = format(new Date(), SDF_YYYY_MM_DD);
//        }

        return format(new Date(), SDF_YYYY_MM_DD);
    }

    public static void setToday(String today) {
        DateUtils.today = today;
    }

    public static String getCurrentTime() {

        return format(new Date());
    }

    /**
     * 获取偏移天数后的日期，零时正点
     * @param offset
     * @return
     */
    public static Date getDay(int offset) {
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(new Date());
        calendar.add(Calendar.DATE, offset);
        return parse(format(calendar.getTime(), SDF_YYYY_MM_DD), SDF_YYYY_MM_DD);
    }

    public static Date getDay(int offsetType, int offset) {
        Calendar calendar = Calendar.getInstance();
        calendar.add(offsetType, offset);
        return calendar.getTime();
    }

    public static Date getDay(Date date, int offsetType, int offset) {
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(date);
        calendar.add(offsetType, offset);
        return calendar.getTime();
    }

     public static String getDayStr(int offset) {
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(new Date());
        calendar.add(Calendar.DATE, offset);
        return format(calendar.getTime(), SDF_YYYY_MM_DD);
    }

    public static String getDayStr(int offsetType, int offset) {
        Calendar calendar = Calendar.getInstance();
        calendar.add(offsetType, offset);
        return DateFormatUtils.format(calendar.getTime(), "yyyy-MM-dd");
    }

    public static String getDayStr(Date date, int offsetType, int offset) {
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(date);
        calendar.add(offsetType, offset);
        return DateFormatUtils.format(calendar.getTime(), "yyyy-MM-dd");
    }

    public static Date getDayTime(int offset) {
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(new Date());
        calendar.add(Calendar.DATE, offset);
        return parse(format(calendar.getTime(), SDF_YYYY_MM_DD_HH_MM_SS), SDF_YYYY_MM_DD_HH_MM_SS);
    }

    public static String getDayTimeStr(int offset) {
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(new Date());
        calendar.add(Calendar.DATE, offset);
        return format(calendar.getTime(), SDF_YYYY_MM_DD_HH_MM_SS);
    }

    public static int compare(Date currentDate , Date compareDate, int calendarType) {
        Calendar current = DateUtils.toCalendar(currentDate);
        Calendar compare = DateUtils.toCalendar(compareDate);
        long gap = current.getTimeInMillis() - compare.getTimeInMillis();
        long dividend = 1000;
        switch (calendarType){
            case Calendar.YEAR : {
                return current.get(calendarType) - compare.get(calendarType);
            }
            case Calendar.MONTH : {
                int year = current.get(Calendar.YEAR) - compare.get(Calendar.YEAR);
                return year * 12 + current.get(calendarType) - compare.get(calendarType);
            }
            case Calendar.DAY_OF_YEAR : {
                dividend *= 60 * 60 * 24;
                break;
            }
            case Calendar.HOUR : {
                dividend *= 60 * 60;
                break;
            }
            case Calendar.MINUTE : {
                dividend *= 60;
                break;
            }
            case Calendar.SECOND : {
                break;
            }
             default:{
                 throw new RuntimeException("尚未维护该类型对比值");
            }
        }
        //向上取整
        return (int)(gap / dividend);
    }

    /**
     * 时间差
     * @param endDate
     * @param nowDate
     * @return
     */
    public static String getTimeDifferenceDHMS(Date endDate, Date nowDate) {

        long nd = 1000 * 24 * 60 * 60;
        long nh = 1000 * 60 * 60;
        long nm = 1000 * 60;
        long diff = endDate.getTime() - nowDate.getTime();
        long day = diff / nd;
        long hour = diff % nd / nh;
        long min = diff % nd % nh / nm;
        long seconds = diff % nd % nh % nm / 1000;
        return day + "天" + hour + "小时" + min + "分钟" + seconds + "秒";
    }

    public static String getTimeDifferenceHMS(Date endDate, Date nowDate) {

        long nd = 1000 * 24 * 60 * 60;
        long nh = 1000 * 60 * 60;
        long nm = 1000 * 60;
        long diff = endDate.getTime() - nowDate.getTime();
        long hour = diff % nd / nh;
        long min = diff % nd % nh / nm;
        long seconds = diff % nd % nh % nm / 1000;
        return hour + "小时" + min + "分钟" + seconds + "秒";
    }

    /**
     * 上月的第一天
     * @return
     */
    public static String getBeforeMonthFirstDay(){
        SimpleDateFormat format = new SimpleDateFormat("yyyy-MM-dd");
        Calendar c =Calendar.getInstance();
        c.add(Calendar.MONTH, -1);
        c.set(Calendar.DAY_OF_MONTH,1);
        return format.format(c.getTime()) + " " + "00:00:00";
    }

    /**
     * 上月最后一天
     * @return
     */
    public static String getBeforeMonthLastDay(){
        SimpleDateFormat format = new SimpleDateFormat("yyyy-MM-dd");
        Calendar c = Calendar.getInstance();
        c.set(Calendar.DAY_OF_MONTH,0);//设置为1号,当前日期既为本月第一天
        return format.format(c.getTime()) + " " + "00:00:00";
    }

    /**
     * 当前月第一天
     * @return
     */
    public static String getCurrentMonthFirstDay(){
        SimpleDateFormat format = new SimpleDateFormat("yyyy-MM-dd");
        Calendar c = Calendar.getInstance();
        c.add(Calendar.MONTH, 0);
        c.set(Calendar.DAY_OF_MONTH,1);
        return format.format(c.getTime()) + " " + "00:00:00";
    }

    public static String getFirstDayOfTheMonth(Date date){
        return getFirstDayOfMonth(date,0);
    }

    public static String getFirstDayOfMonth(Date date,int month){
        SimpleDateFormat format = new SimpleDateFormat("yyyy-MM-dd");
        Calendar c = Calendar.getInstance();
        c.setTime(date);
        c.add(Calendar.MONTH, month);
        c.set(Calendar.DAY_OF_MONTH,1);
        return format.format(c.getTime()) + " " + "00:00:00";
    }

    /**
     * 当前月第一天
     * @return
     */
    public static String getCurrentMonthLastDay(){
        SimpleDateFormat format = new SimpleDateFormat("yyyy-MM-dd");
        Calendar c = Calendar.getInstance();
        c.set(Calendar.DAY_OF_MONTH, c.getActualMaximum(Calendar.DAY_OF_MONTH));
        return format.format(c.getTime()) + " " + "00:00:00";
    }

    /**
     * 获取传入时间的零点
     */
    public static Date getZeroDate(Date date) {
        Calendar c = Calendar.getInstance();
        c.setTime(date);
        c.set(Calendar.HOUR_OF_DAY, 0);
        c.set(Calendar.MINUTE, 0);
        c.set(Calendar.SECOND, 0);
        c.set(Calendar.MILLISECOND, 0);
        return c.getTime();
    }

    public static Date geLastWeekMonday() {
        Calendar cal = Calendar.getInstance();
        cal.setTime(getThisWeekMonday());
        cal.add(Calendar.DATE, -7);
        return cal.getTime();
    }

    public static Date getThisWeekMonday() {
        Calendar cal = Calendar.getInstance();
        cal.setTime(DateUtils.getDay(0));
        // 获得当前日期是一个星期的第几天
        int dayWeek = cal.get(Calendar.DAY_OF_WEEK);
        if (1 == dayWeek) {
            cal.add(Calendar.DAY_OF_MONTH, -1);
        }
        // 设置一个星期的第一天，按中国的习惯一个星期的第一天是星期一
        cal.setFirstDayOfWeek(Calendar.MONDAY);
        // 获得当前日期是一个星期的第几天
        int day = cal.get(Calendar.DAY_OF_WEEK);
        // 根据日历的规则，给当前日期减去星期几与一个星期第一天的差值
        cal.add(Calendar.DATE, cal.getFirstDayOfWeek() - day);
        return cal.getTime();
    }

    public static Date getLastMonthFirstDay(){
        Calendar cal = Calendar.getInstance();
        cal.setTime(DateUtils.getDay(0));
        cal.add(Calendar.MONTH, -1);
        cal.set(Calendar.DAY_OF_MONTH,
                cal.getActualMinimum(Calendar.DAY_OF_MONTH));
        return cal.getTime();

    }

    public static Date getThisMonthFirstDay(){
        Calendar cal = Calendar.getInstance();
        cal.setTime(DateUtils.getDay(0));
        cal.set(Calendar.DAY_OF_MONTH,
                cal.getActualMinimum(Calendar.DAY_OF_MONTH));
        return cal.getTime();
    }
}
