package com.afanticar.afantiopenapi.controller;

import cn.hutool.core.date.DateUtil;
import cn.hutool.core.util.StrUtil;
import com.afanticar.activity.client.IActivityManagerApiClient;
import com.afanticar.activity.domain.param.*;
import com.afanticar.activity.domain.vo.ApiActivityInfoVO;
import com.afanticar.afantiopenapi.constant.Constant;
import com.afanticar.afantiopenapi.model.BaseResponse;
import com.afanticar.afantiopenapi.model.dto.FawActivityDTO;
import com.afanticar.afantiopenapi.model.vo.*;
import com.afanticar.afantiopenapi.service.BaseService;
import com.afanticar.afantiopenapi.service.FawOpenService;
import com.afanticar.common.base.domain.BasePage;
import com.afanticar.common.core.api.CommonResult;
import com.alibaba.fastjson.JSONObject;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletRequest;
import javax.validation.constraints.Max;
import javax.validation.constraints.NotNull;
import java.util.Date;

/**
 * @author chin
 * @contact chenyan@afanticar.com
 * @since 2023/4/26/026
 */
@Api("红旗数据接口")
@RestController
@RequestMapping("/faw")
@Slf4j
@Validated
public class FawOpenController extends BaseController {

    @Autowired
    FawOpenService fawOpenService;

    @Autowired
    private IActivityManagerApiClient activityManagerApiClient;

    @Autowired
    BaseService baseService;

    @Value("${faw.client}")
    String client;

    @Value("${faw.limit.period:60}")
    Integer limitPeriod;

    @Value("${faw.limit.threshold:5}")
    Integer limit;

    /**
     * 活动创建，  1天不能超过100个 -uat环境
     */
    @Value("${faw.activity.create.limit.period:86400}")
    Integer activityCreateLimitPeriod;

    @Value("${faw.activity.create.limit.threshold:100}")
    Integer activityCreateLimit;

    /**
     * 活动查询修改、删除60秒内不能超过120次
     */
    @Value("${faw.activity.query.limit.period:60}")
    Integer activityCurdLimitPeriod;

    @Value("${faw.activity.query.limit.threshold:120}")
    Integer activityCurdLimit;

    /**
     * 红旗主体id
     */
    @Value("${faw.principalId:869752931296989184}")
    String fawPrincipalId;

    static String dataType = "faw";

    static String monthly = "monthly", quarter = "quarter";

    @ApiOperation(value = "视频数据信息", notes = "视频数据", produces = "application/json")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "publish_date_start", value = "开始时间，格式：2023-02-01 00:00:00", required = true, paramType = "path", dataType = "Date"),
            @ApiImplicitParam(name = "publish_date_end", value = "结束时间，格式：2023-03-01 00:00:00", required = true, paramType = "path", dataType = "Date"),
            @ApiImplicitParam(name = "type", value = "数据类型，monthly-月度表，quarter-季度表", required = true, paramType = "path", dataType = "String"),
            @ApiImplicitParam(name = "current_page", value = "页码", paramType = "path", dataType = "Integer"),
            @ApiImplicitParam(name = "page_size", value = "页数", paramType = "path", dataType = "Integer"),
    })
    @GetMapping("/aweme")
    public BaseResponse<BasePageVO<HongqiAwemeIncentiveVO>> getAwemeData(HttpServletRequest request, @Max(value = 500,message = "单页数据page_size过大") @RequestParam(name = "page_size", defaultValue = "100", required = false) Integer pageSize,
                                                                           @RequestParam(name = "current_page", defaultValue = "1", required = false) Integer pageNum,
                                                                         @NotNull(message = "publish_date_start不能为空") @RequestParam(name = "publish_date_start", required = false) String publishDateStartStr,
                                                                         @NotNull(message = "publish_date_end不能为空") @RequestParam(name = "publish_date_end", required = false) String publishDateEndStr,
                                                                         @NotNull(message = "type不能为空") @RequestParam(name = "type", required = false) String type) {
        Date publishDateEnd = parseDate(publishDateEndStr), publishDateStart = parseDate(publishDateStartStr);
        if (publishDateEnd == null || publishDateStart == null) {
            return error("90001", "publish_date_end或publish_date_start时间格式不合法");
        }
        log.info("AAAAAAAAA:" + limit);
        publishDateEnd = endTime(publishDateEnd);
        if (!type.equals(quarter) && !type.equals(monthly)) {
            return error("90001", "type传参异常，未开放数据类型");
        }
        return success(fawOpenService.awemes(publishDateStart, publishDateEnd, type, pageNum, pageSize));
    }

    @ApiOperation(value = "直播数据信息", notes = "直播数据", produces = "application/json")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "publish_date_start", value = "开始时间，格式：2023-02-01", required = true, paramType = "path", dataType = "Date"),
            @ApiImplicitParam(name = "publish_date_end", value = "结束时间，格式：2023-03-01", required = true, paramType = "path", dataType = "Date"),
            @ApiImplicitParam(name = "type", value = "数据类型，monthly-月度表，quarter-季度表", required = true, paramType = "path", dataType = "String"),
            @ApiImplicitParam(name = "current_page", value = "页码", paramType = "path", dataType = "Integer"),
            @ApiImplicitParam(name = "page_size", value = "页数", paramType = "path", dataType = "Integer"),
    })
    @GetMapping("/live")
    public BaseResponse<BasePageVO<HongqiLiveIncentiveVO>> getLiveData(HttpServletRequest request, @Max(value = 500,message = "单页数据page_size过大") @RequestParam(name = "page_size", defaultValue = "100", required = false) Integer pageSize,
                                                                       @RequestParam(name = "current_page", defaultValue = "1", required = false) Integer pageNum,
                                                                       @NotNull(message = "publish_date_start不能为空") @RequestParam(name = "publish_date_start", required = false) String publishDateStartStr,
                                                                       @NotNull(message = "publish_date_end不能为空") @RequestParam(name = "publish_date_end", required = false) String publishDateEndStr,
                                                                       @NotNull(message = "type不能为空") @RequestParam(name = "type", required = false) String type) {
        Date publishDateEnd = parseDate(publishDateEndStr), publishDateStart = parseDate(publishDateStartStr);
        if (publishDateEnd == null || publishDateStart == null) {
            return error("90001", "publish_date_end或者publish_date_start时间格式不合法");
        }
        publishDateEnd = endTime(publishDateEnd);
        if (!type.equals(quarter) && !type.equals(monthly)) {
            return error("90001", "type传参异常，未开放数据类型！");
        }
        return success(fawOpenService.lives(publishDateStart, publishDateEnd, type, pageNum, pageSize));
    }

    private Date endTime(Date endDate) {
        return DateUtil.endOfDay(endDate);
    }

    @ApiOperation(value = "抖音活动视频数据信息", notes = "抖音活动视频数据", produces = "application/json")
    @PostMapping("/activity/douyin/aweme/list")
    public BaseResponse<BasePageVO<HongqiActivityAwemeVO>> getDouyinActivityAwemeData(@RequestBody @Validated FawActivityDTO dto) {
        BaseResponse check = baseService.pageApiLimit(dataType, limit, limitPeriod);
        if (check != null) {
            return check;
        }
        check = activityRequestDTO(dto);
        if (check != null) {
            return check;
        }
        return success(fawOpenService.douyinActivityAwemes(dto));
    }

    @ApiOperation(value = "抖音活动直播数据信息", notes = "抖音活动直播数据", produces = "application/json")
    @PostMapping("/activity/douyin/live/list")
    public BaseResponse<BasePageVO<HongqiActivityLiveVO>> getDouyinActivityLiveData(@RequestBody @Validated FawActivityDTO dto) {
        BaseResponse check = baseService.pageApiLimit(dataType, limit, limitPeriod);
        if (check != null) {
            return check;
        }
        check = activityRequestDTO(dto);
        if (check != null) {
            return check;
        }
        return success(fawOpenService.douyinActivityLives(dto));
    }

    @ApiOperation(value = "快手活动视频数据信息", notes = "快手活动视频数据", produces = "application/json")
    @PostMapping("/activity/kuaishou/aweme/list")
    public BaseResponse<BasePageVO<HongqiActivityAwemeVO>> getKuaishouActivityAwemeData(@RequestBody @Validated FawActivityDTO dto) {
        BaseResponse check = baseService.pageApiLimit(dataType, limit, limitPeriod);
        if (check != null) {
            return check;
        }
        check = activityRequestDTO(dto);
        if (check != null) {
            return check;
        }
        return success(fawOpenService.kuaishouActivityAwemes(dto));
    }

    @ApiOperation(value = "快手活动直播数据信息", notes = "快手活动直播数据", produces = "application/json")
    @PostMapping("/activity/kuaishou/live/list")
    public BaseResponse<BasePageVO<HongqiActivityLiveVO>> getKuaishouActivityLiveData(@RequestBody @Validated FawActivityDTO dto) {
        BaseResponse check = baseService.pageApiLimit(dataType, limit, limitPeriod);
        if (check != null) {
            return check;
        }
        check = activityRequestDTO(dto);
        if (check != null) {
            return check;
        }
        return success(fawOpenService.kuaishouActivityLives(dto));
    }

    @ApiOperation(value = "创建活动", notes = "创建活动", produces = "application/json")
    @PostMapping("/activity/create")
    public BaseResponse<String> activityCreate(@RequestBody ApiActivityCreateParam param) {
        log.info("红旗活动-活动创建req={}", JSONObject.toJSONString(param));
        BaseResponse check = baseService.baseLimit(dataType+":activity_create", activityCreateLimit, activityCreateLimitPeriod);
        if (check != null) {
            return check;
        }
        if(StrUtil.isEmpty(param.getActor())){
            param.setActor("external-system");
        }
        param.setPrincipalId(fawPrincipalId);
        CommonResult<String> res = activityManagerApiClient.activityCreate(param);
        if(Constant.SUCCESS_CODE.equals(res.getCode().toString())){
            return success(res.getData());
        }else{
            return error(res.getCode().toString(),res.getMessage());
        }
    }

    @ApiOperation(value = "活动更新", notes = "活动更新", produces = "application/json")
    @PostMapping("/activity/update")
    public BaseResponse<String> activityUpdate(@RequestBody ApiActivityUpdateParam param) {
        log.info("红旗活动-活动更新req={}", JSONObject.toJSONString(param));
        BaseResponse check = baseService.baseLimit(dataType+":activity_curd", activityCurdLimit, activityCurdLimitPeriod);
        if (check != null) {
            return check;
        }
        if(StrUtil.isEmpty(param.getActor())){
            param.setActor("external-system");
        }
        param.setPrincipalId(fawPrincipalId);
        CommonResult<String> res = activityManagerApiClient.activityUpdate(param);
        if(Constant.SUCCESS_CODE.equals(res.getCode().toString())){
            return success(res.getData());
        }else{
            return error(res.getCode().toString(),res.getMessage());
        }
    }

    @ApiOperation(value = "删除活动", notes = "删除活动", produces = "application/json")
    @PostMapping("/activity/delete")
    public BaseResponse<String> activityDelete(@RequestBody ApiActivityDeletedParam param) {
        log.info("红旗活动-删除活动req={}", JSONObject.toJSONString(param));
        BaseResponse check = baseService.baseLimit(dataType+":activity_curd", activityCurdLimit, activityCurdLimitPeriod);
        if (check != null) {
            return check;
        }
        if(StrUtil.isEmpty(param.getActor())){
            param.setActor("external-system");
        }
        param.setPrincipalId(fawPrincipalId);
        CommonResult<String> res = activityManagerApiClient.activityDeleted(param);
        if(Constant.SUCCESS_CODE.equals(res.getCode().toString())){
            return success();
        }else{
            return error(res.getCode().toString(),res.getMessage());
        }
    }

    @ApiOperation(value = "活动列表", notes = "活动列表", produces = "application/json")
    @PostMapping("/activity/list")
    public BaseResponse<BasePageVO<ApiActivityInfoVO>> activityList(@RequestBody ApiActivityListParam param) {
        log.info("红旗活动-活动列表req={}", JSONObject.toJSONString(param));
        BaseResponse check = baseService.baseLimit(dataType+":activity_curd", activityCurdLimit, activityCurdLimitPeriod);
        if (check != null) {
            return check;
        }
        if(StrUtil.isEmpty(param.getActor())){
            param.setActor("external-system");
        }
        param.setPrincipalId(fawPrincipalId);
        CommonResult<BasePage<ApiActivityInfoVO>> res = activityManagerApiClient.activityList(param);
        if(Constant.SUCCESS_CODE.equals(res.getCode().toString())){
            BasePageVO basePageVO = new BasePageVO();
            basePageVO.setPages(res.getData().getTotalPage().intValue());
            basePageVO.setTotal(res.getData().getTotal().intValue());
            basePageVO.setCurrent(res.getData().getPage().intValue());
            basePageVO.setRecords(res.getData().getList());
            basePageVO.setSize(res.getData().getSize().intValue());
            return success(basePageVO);
        }else{
            return error(res.getCode().toString(),res.getMessage());
        }
    }

    @ApiOperation(value = "活动详情", notes = "活动详情")
    @GetMapping("/activity/info")
    public BaseResponse<ApiActivityInfoVO> activityList(@RequestBody(required = false) ApiActivityInfoParam param,
                                                        @RequestParam(value = "activity_id",required = false) String activityId,
                                                        @RequestParam(value = "actor",required = false) String actor) {
        log.info("红旗活动-活动详情req={},activityId={},actor={}", JSONObject.toJSONString(param),activityId,actor);
        BaseResponse check = baseService.baseLimit(dataType+":activity_curd", activityCurdLimit, activityCurdLimitPeriod);
        if (check != null) {
            return check;
        }
        boolean paramIsEmpty = param == null || StrUtil.isEmpty(param.getActivityId());
        if(paramIsEmpty && StrUtil.isEmpty(activityId)){
            return error("400","活动id不能为空");
        }
        if(param == null){
            param = new ApiActivityInfoParam();
            param.setActivityId(activityId);
            param.setActor(actor);
        }
        if(StrUtil.isEmpty(param.getActor())){
            param.setActor("external-system");
        }
        param.setPrincipalId(fawPrincipalId);
        CommonResult<ApiActivityInfoVO> res = activityManagerApiClient.activityInfo(param);
        if(Constant.SUCCESS_CODE.equals(res.getCode().toString())){
            return success(res.getData());
        }else{
            return error(res.getCode().toString(),res.getMessage());
        }
    }
}
