package com.afanticar.afantiopenapi.controller;

import com.afanticar.afantiopenapi.constant.PrincipalEnum;
import com.afanticar.afantiopenapi.model.BaseResponse;
import com.afanticar.afantiopenapi.model.vo.BasePageVO;
import com.afanticar.afantiopenapi.model.vo.BydXiaoHongShuAuthorDayVO;
import com.afanticar.afantiopenapi.model.vo.BydXiaoHongShuMemberDayVO;
import com.afanticar.afantiopenapi.model.vo.BydXiaoHongShuNoteDayVO;
import com.afanticar.afantiopenapi.service.BaseService;
import com.afanticar.afantiopenapi.service.BydOpenService;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;
import org.springframework.lang.NonNullApi;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.validation.constraints.Max;
import javax.validation.constraints.NotNull;
import java.time.LocalDate;
import java.time.format.DateTimeParseException;
import java.util.Arrays;
import java.util.Optional;

@Api("比亚迪数据接口")
@RequestMapping("/byd/xhs/")
@RestController
@Slf4j
@Validated
public class BydOpenController extends BaseController{

    @Resource
    BydOpenService bydOpenService;

    @Resource
    BaseService baseService;

    static String dataType = "byd";

    @Value("${faw.limit.period:60}")
    Integer limitPeriod;

    @Value("${faw.limit.threshold:5}")
    Integer limit;

    @ApiOperation(value = "门店日数据", notes = "门店日数据", produces = "application/json")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "principal_code", value = "品牌编码", required = true, paramType = "path", dataType = "String"),
            @ApiImplicitParam(name = "stats_start", value = "开始时间，格式：2023-02-01", required = true, paramType = "path", dataType = "Date"),
            @ApiImplicitParam(name = "stats_end_date", value = "结束时间，格式：2023-03-01", required = true, paramType = "path", dataType = "Date"),
            @ApiImplicitParam(name = "page_size", value = "页数", paramType = "path", dataType = "Integer"),
            @ApiImplicitParam(name = "current_page", value = "页码", paramType = "path", dataType = "Integer"),
    })
    @GetMapping("/member/day")
    public BaseResponse<BasePageVO<BydXiaoHongShuMemberDayVO>> queryMemberDaySnap(
            HttpServletRequest request,
            @NotNull(message = "principal_code不能为空") @RequestParam(name = "principal_code") String principalCode,
            @NotNull(message = "stats_start_date不能为空") @RequestParam(name = "stats_start_date") LocalDate statsStart,
            @NotNull(message = "stats_end_date不能为空") @RequestParam(name = "stats_end_date") LocalDate statsEnd,
            @Max(value = 500, message = "单页数据page_size过大") @RequestParam(name = "page_size", defaultValue = "100", required = false) Integer pageSize,
            @RequestParam(name = "current_page", defaultValue = "1", required = false) Integer currentPage
    ){
        PrincipalEnum principal = PrincipalEnum.getPrincipal(principalCode).orElse(PrincipalEnum.BYD_WC);
        return  success(bydOpenService.fetchMemberDaySnap(principal, statsStart, statsEnd, currentPage, pageSize));
    }


    @ApiOperation(value = "主播日数据", notes = "主播日数据", produces = "application/json")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "principal_code", value = "品牌编码", required = true, paramType = "path", dataType = "String"),
            @ApiImplicitParam(name = "stats_start_date", value = "开始时间，格式：2023-02-01", required = true, paramType = "path", dataType = "Date"),
            @ApiImplicitParam(name = "stats_end_date", value = "结束时间，格式：2023-03-01", required = true, paramType = "path", dataType = "Date"),
            @ApiImplicitParam(name = "page_size", value = "页数", paramType = "path", dataType = "Integer"),
            @ApiImplicitParam(name = "current_page", value = "页码", paramType = "path", dataType = "Integer"),
    })
    @GetMapping("/author/day")
    public BaseResponse<BasePageVO<BydXiaoHongShuAuthorDayVO>> queryAuthorDaySnap(
            HttpServletRequest request,
            @NotNull(message = "principal_code不能为空") @RequestParam(name = "principal_code") String principalCode,
            @NotNull(message = "stats_start_date不能为空") @RequestParam(name = "stats_start_date") LocalDate statsStart,
            @NotNull(message = "stats_end_date不能为空") @RequestParam(name = "stats_end_date") LocalDate statsEnd,
            @Max(value = 500, message = "单页数据page_size过大") @RequestParam(name = "page_size", defaultValue = "100", required = false) Integer pageSize,
            @RequestParam(name = "current_page", defaultValue = "1", required = false) Integer currentPage
    ){
        PrincipalEnum principal = PrincipalEnum.getPrincipal(principalCode).orElse(PrincipalEnum.BYD_WC);
        return success(bydOpenService.fetchAuthorDaySnap(principal, statsStart, statsEnd, currentPage, pageSize));
    }


    @ApiOperation(value = "笔记日数据", notes = "笔记日数据", produces = "application/json")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "principal_code", value = "品牌编码", required = true, paramType = "path", dataType = "String"),
            @ApiImplicitParam(name = "stats_start_date", value = "开始时间，格式：2023-02-01", required = true, paramType = "path", dataType = "Date"),
            @ApiImplicitParam(name = "stats_end_date", value = "结束时间，格式：2023-03-01", required = true, paramType = "path", dataType = "Date"),
            @ApiImplicitParam(name = "page_size", value = "页数", paramType = "path", dataType = "Integer"),
            @ApiImplicitParam(name = "current_page", value = "页码", paramType = "path", dataType = "Integer"),
    })
    @GetMapping("/note/day")
    public BaseResponse<BasePageVO<BydXiaoHongShuNoteDayVO>> queryNoteDaySnap(
            HttpServletRequest request,
            @NotNull(message = "principal_code不能为空") @RequestParam(name = "principal_code") String principalCode,
            @NotNull(message = "stats_date_start不能为空") @RequestParam(name = "stats_start_date") LocalDate statsStart,
            @NotNull(message = "stats_date_end不能为空") @RequestParam(name = "stats_end_date") LocalDate statsEnd,
            @RequestParam(name = "publish_start_date", required = false) LocalDate pubStart,
            @RequestParam(name = "publish_end_date", required = false) LocalDate pubEnd,
            @Max(value = 500, message = "单页数据page_size过大") @RequestParam(name = "page_size", defaultValue = "100", required = false) Integer pageSize,
            @RequestParam(name = "current_page", defaultValue = "1", required = false) Integer currentPage
    ){
        PrincipalEnum principal = PrincipalEnum.getPrincipal(principalCode).orElse(PrincipalEnum.BYD_WC);
        return success(bydOpenService.fetchNoteDaySnap(principal, statsStart, statsEnd, pubStart, pubEnd, currentPage, pageSize));
    }

    @Override
    public boolean preHandle(HttpServletRequest request, HttpServletResponse response, Object handler) {
        BaseResponse<Object> check = baseService.pageApiLimit(dataType, limit, limitPeriod);
        if (check != null) {
            setResponse(response, Integer.parseInt(check.getCode()), check.getMessage());
            return false;
        }

        String requestUri = request.getRequestURI();
        if (requestUri.startsWith("/byd")) {
            String principalCode = Optional.ofNullable(request.getParameter("principal_code")).orElse("");
            Optional<PrincipalEnum> principal = PrincipalEnum.getPrincipal(principalCode);
            if (!principal.isPresent()) {
                setResponse(response, 60002, "principal_code必须为" + Arrays.toString(PrincipalEnum.values()) +"其中之一.");
                return false;
            }

            Optional<String> statsStartStr = Optional.ofNullable(request.getParameter("stats_start_date"));
            Optional<String> statsEndStr = Optional.ofNullable(request.getParameter("stats_start_date"));
            if (statsStartStr.isPresent() && statsEndStr.isPresent()) {
                try {
                    LocalDate statsStart = LocalDate.parse(statsStartStr.get());
                    LocalDate statsEnd = LocalDate.parse(statsEndStr.get());
                    if (statsStart.isAfter(statsEnd)) {
                        setResponse(response, 60003, "统计开始日期不能大于结束日期.");
                        return false;
                    } else if (statsEnd.isAfter(LocalDate.now().minusDays(1))){
                        setResponse(response, 60003, "统计结束日期不能大于昨天.");
                        return false;
                    } else if (statsStart.plusDays(5).isBefore(statsEnd)) {
                        setResponse(response, 60003, "单次查询统计日期窗口不能超过5天.");
                        return false;
                    }
                } catch (DateTimeParseException e) {
                    setResponse(response, 400, "统计开始日期或结束日期格式不正确.");
                    return false;
                }
            } else {
                setResponse(response, 400, "统计开始日期和结束日期为必传参数.");
                return false;
            }

            Optional<String> pubStartStr = Optional.ofNullable(request.getParameter("publish_start_date"));
            Optional<String> pubEndStr = Optional.ofNullable(request.getParameter("publish_start_date"));
            if (pubStartStr.isPresent() && pubEndStr.isPresent()) {
                try {
                    LocalDate pubStart = LocalDate.parse(pubStartStr.get());
                    LocalDate pubEnd = LocalDate.parse(pubEndStr.get());
                    if (pubStart.isAfter(pubEnd)) {
                        setResponse(response, 60003, "发布开始日期不能大于结束日期.");
                        return false;
                    } else if (pubEnd.isAfter(LocalDate.now().minusDays(1))) {
                        setResponse(response, 60003, "发布结束日期不能大于昨天.");
                        return false;
                    }
                } catch (DateTimeParseException e) {
                    setResponse(response, 400, "发布开始日期或结束日期格式不正确.");
                    return false;
                }
            } else if (pubStartStr.isPresent() || pubEndStr.isPresent()) {
                setResponse(response, 400, "统计开始日期和结束日期参数需要同时提供或者不提供.");
                return false;
            }

        }
        return true;
    }

}
