package com.afanticar.afantiopenapi.controller;

import cn.hutool.core.date.DateUtil;
import com.afanticar.afantiopenapi.model.BaseResponse;
import com.afanticar.afantiopenapi.model.dto.FawActivityDTO;
import com.afanticar.afantiopenapi.model.vo.*;
import com.afanticar.afantiopenapi.service.BaseService;
import com.afanticar.afantiopenapi.service.FawOpenService;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletRequest;
import javax.validation.constraints.Max;
import javax.validation.constraints.NotNull;
import java.util.Date;

/**
 * @author chin
 * @contact chenyan@afanticar.com
 * @since 2023/4/26/026
 */
@Api("红旗数据接口")
@RestController
@RequestMapping("/faw")
@Slf4j
@Validated
public class FawOpenController extends BaseController {

    @Autowired
    FawOpenService fawOpenService;

    @Autowired
    BaseService baseService;

    @Value("${faw.client}")
    String client;

    @Value("${faw.limit.period:60}")
    Integer limitPeriod;

    @Value("${faw.limit.threshold:5}")
    Integer limit;

    static String dataType = "faw";

    static String monthly = "monthly", quarter = "quarter";

    @ApiOperation(value = "视频数据信息", notes = "视频数据", produces = "application/json")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "publish_date_start", value = "开始时间，格式：2023-02-01 00:00:00", required = true, paramType = "path", dataType = "Date"),
            @ApiImplicitParam(name = "publish_date_end", value = "结束时间，格式：2023-03-01 00:00:00", required = true, paramType = "path", dataType = "Date"),
            @ApiImplicitParam(name = "type", value = "数据类型，monthly-月度表，quarter-季度表", required = true, paramType = "path", dataType = "String"),
            @ApiImplicitParam(name = "current_page", value = "页码", paramType = "path", dataType = "Integer"),
            @ApiImplicitParam(name = "page_size", value = "页数", paramType = "path", dataType = "Integer"),
    })
    @GetMapping("/aweme")
    public BaseResponse<BasePageVO<HongqiAwemeIncentiveVO>> getAwemeData(HttpServletRequest request, @Max(value = 500,message = "单页数据page_size过大") @RequestParam(name = "page_size", defaultValue = "100", required = false) Integer pageSize,
                                                                           @RequestParam(name = "current_page", defaultValue = "1", required = false) Integer pageNum,
                                                                         @NotNull(message = "publish_date_start不能为空") @RequestParam(name = "publish_date_start", required = false) String publishDateStartStr,
                                                                         @NotNull(message = "publish_date_end不能为空") @RequestParam(name = "publish_date_end", required = false) String publishDateEndStr,
                                                                         @NotNull(message = "type不能为空") @RequestParam(name = "type", required = false) String type) {
        Date publishDateEnd = parseDate(publishDateEndStr), publishDateStart = parseDate(publishDateStartStr);
        if (publishDateEnd == null || publishDateStart == null) {
            return error("90001", "publish_date_end或publish_date_start时间格式不合法");
        }
        log.info("AAAAAAAAA:" + limit);
        publishDateEnd = endTime(publishDateEnd);
        if (!type.equals(quarter) && !type.equals(monthly)) {
            return error("90001", "type传参异常，未开放数据类型");
        }
        return success(fawOpenService.awemes(publishDateStart, publishDateEnd, type, pageNum, pageSize));
    }

    @ApiOperation(value = "直播数据信息", notes = "直播数据", produces = "application/json")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "publish_date_start", value = "开始时间，格式：2023-02-01", required = true, paramType = "path", dataType = "Date"),
            @ApiImplicitParam(name = "publish_date_end", value = "结束时间，格式：2023-03-01", required = true, paramType = "path", dataType = "Date"),
            @ApiImplicitParam(name = "type", value = "数据类型，monthly-月度表，quarter-季度表", required = true, paramType = "path", dataType = "String"),
            @ApiImplicitParam(name = "current_page", value = "页码", paramType = "path", dataType = "Integer"),
            @ApiImplicitParam(name = "page_size", value = "页数", paramType = "path", dataType = "Integer"),
    })
    @GetMapping("/live")
    public BaseResponse<BasePageVO<HongqiLiveIncentiveVO>> getLiveData(HttpServletRequest request, @Max(value = 500,message = "单页数据page_size过大") @RequestParam(name = "page_size", defaultValue = "100", required = false) Integer pageSize,
                                                                       @RequestParam(name = "current_page", defaultValue = "1", required = false) Integer pageNum,
                                                                       @NotNull(message = "publish_date_start不能为空") @RequestParam(name = "publish_date_start", required = false) String publishDateStartStr,
                                                                       @NotNull(message = "publish_date_end不能为空") @RequestParam(name = "publish_date_end", required = false) String publishDateEndStr,
                                                                       @NotNull(message = "type不能为空") @RequestParam(name = "type", required = false) String type) {
        Date publishDateEnd = parseDate(publishDateEndStr), publishDateStart = parseDate(publishDateStartStr);
        if (publishDateEnd == null || publishDateStart == null) {
            return error("90001", "publish_date_end或者publish_date_start时间格式不合法");
        }
        publishDateEnd = endTime(publishDateEnd);
        if (!type.equals(quarter) && !type.equals(monthly)) {
            return error("90001", "type传参异常，未开放数据类型！");
        }
        return success(fawOpenService.lives(publishDateStart, publishDateEnd, type, pageNum, pageSize));
    }

    private Date endTime(Date endDate) {
        return DateUtil.endOfDay(endDate);
    }

    @ApiOperation(value = "抖音活动视频数据信息", notes = "抖音活动视频数据", produces = "application/json")
    @PostMapping("/activity/douyin/aweme/list")
    public BaseResponse<BasePageVO<HongqiActivityAwemeVO>> getDouyinActivityAwemeData(@RequestBody @Validated FawActivityDTO dto) {
        BaseResponse check = baseService.baseLimit(dataType, limit, limitPeriod);
        if (check != null) {
            return check;
        }
        check = activityRequestDTO(dto);
        if (check != null) {
            return check;
        }
        return success(fawOpenService.douyinActivityAwemes(dto));
    }

    @ApiOperation(value = "抖音活动直播数据信息", notes = "抖音活动直播数据", produces = "application/json")
    @PostMapping("/activity/douyin/live/list")
    public BaseResponse<BasePageVO<HongqiActivityLiveVO>> getDouyinActivityLiveData(@RequestBody @Validated FawActivityDTO dto) {
        BaseResponse check = baseService.baseLimit(dataType, limit, limitPeriod);
        if (check != null) {
            return check;
        }
        check = activityRequestDTO(dto);
        if (check != null) {
            return check;
        }
        return success(fawOpenService.douyinActivityLives(dto));
    }

    @ApiOperation(value = "快手活动视频数据信息", notes = "快手活动视频数据", produces = "application/json")
    @PostMapping("/activity/kuaishou/aweme/list")
    public BaseResponse<BasePageVO<HongqiActivityAwemeVO>> getKuaishouActivityAwemeData(@RequestBody @Validated FawActivityDTO dto) {
        BaseResponse check = baseService.baseLimit(dataType, limit, limitPeriod);
        if (check != null) {
            return check;
        }
        check = activityRequestDTO(dto);
        if (check != null) {
            return check;
        }
        return success(fawOpenService.kuaishouActivityAwemes(dto));
    }

    @ApiOperation(value = "快手活动直播数据信息", notes = "快手活动直播数据", produces = "application/json")
    @PostMapping("/activity/kuaishou/live/list")
    public BaseResponse<BasePageVO<HongqiActivityLiveVO>> getKuaishouActivityLiveData(@RequestBody @Validated FawActivityDTO dto) {
        BaseResponse check = baseService.baseLimit(dataType, limit, limitPeriod);
        if (check != null) {
            return check;
        }
        check = activityRequestDTO(dto);
        if (check != null) {
            return check;
        }
        return success(fawOpenService.kuaishouActivityLives(dto));
    }

}
