package com.afanticar.afantiopenapi.config;

import com.afanticar.afantiopenapi.constant.ExceptionEnum;
import com.afanticar.afantiopenapi.controller.BaseController;
import com.afanticar.afantiopenapi.model.BaseResponse;
import com.afanticar.common.core.api.ResultCodeEnum;
import com.afanticar.common.core.exception.ApiException;
import org.apache.commons.lang3.exception.ExceptionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.http.converter.HttpMessageNotReadableException;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.RestControllerAdvice;

import javax.servlet.http.HttpServletResponse;
import javax.validation.ConstraintViolationException;
import java.util.StringJoiner;

@RestControllerAdvice
public class GlobalExceptionHandler {

    protected final Logger LOGGER = LoggerFactory.getLogger(this.getClass());

    @ExceptionHandler(ConstraintViolationException.class)
    public BaseResponse processMethodArgumentNotValidException(HttpServletResponse response, ConstraintViolationException e) {
        StringJoiner message = new StringJoiner(";");
        e.getConstraintViolations().forEach(constraintViolation -> message.add(constraintViolation.getMessage()));
        LOGGER.error("参数校验异常 => ", e);
        return handleError(ExceptionEnum.PARAM_INVALID.getCode(), String.join(":", ExceptionEnum.PARAM_INVALID.getMessage(), message.toString()), null);
    }

    @ExceptionHandler(Exception.class)
    public BaseResponse processException(Exception e) {
        LOGGER.error("未知异常 => :{}", ExceptionUtils.getStackTrace(e));
        return handleError(ExceptionEnum.ERROR, e);
    }

    @ExceptionHandler(HttpMessageNotReadableException.class)
    public BaseResponse processHttpMessageNotReadableException(HttpMessageNotReadableException e) {
        LOGGER.error("HttpMessageNotReadableException 调用异常 => :{}", ExceptionUtils.getStackTrace(e));
        return BaseController.error(ResultCodeEnum.VALIDATE_FAILED.getCode().toString(), e.getMessage());
    }

    @ExceptionHandler(ApiException.class)
    public BaseResponse processApiException(ApiException e) {
        LOGGER.error("api 调用异常 => :{}", ExceptionUtils.getStackTrace(e));
        return BaseController.error(e.getCode().toString(), e.getMessage());
    }

    private BaseResponse<Object> handleError(int code, String message, Exception ex) {
        return BaseController.error(code + "", message);
    }

    private BaseResponse<Object> handleError(ExceptionEnum exceptionEnum, Exception ex) {
        // 状态码
        Integer code = exceptionEnum.getCode();
        // 状态码描述
        String message = exceptionEnum.getMessage();
        return handleError(code, message, ex);
    }
}
